<?php
/* --------------------------------------------------------------
   WithdrawalApiRequestParser.php 2020-03-11
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Api\Withdrawal;

use DateTime;
use Exception;
use Gambio\Admin\Withdrawal\SqlCriteria\WithdrawalFilter;
use Gambio\Admin\Withdrawal\SqlCriteria\WithdrawalFilters;
use Gambio\Admin\Withdrawal\SqlCriteria\WithdrawalSorting;
use Gambio\Admin\Withdrawal\SqlCriteria\WithdrawalSortings;
use Gambio\Admin\Withdrawal\SqlCriteria\WithdrawalSqlCriteria;
use Gambio\Admin\Withdrawal\SqlCriteria\WithdrawalSqlPagination;
use Psr\Http\Message\ServerRequestInterface;

/**
 * Class WithdrawalApiRequestParser
 *
 * @package Gambio\Api\Withdrawal
 */
class WithdrawalApiRequestParser
{
    /**
     * @param ServerRequestInterface $request
     *
     * @return int
     */
    public function getPerPageFromRequest(ServerRequestInterface $request): int
    {
        return (int)$request->getQueryParam('per-page', 25);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return int
     */
    public function getPageFromRequest(ServerRequestInterface $request): int
    {
        return (int)$request->getQueryParam('page', 1);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return array
     */
    public function getFieldsFromRequest(ServerRequestInterface $request): array
    {
        $fields = $request->getQueryParam('fields');
        
        return ($fields === null) ? [] : explode(',', $fields);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return WithdrawalSqlPagination
     */
    public function getPaginationFromRequest(ServerRequestInterface $request): WithdrawalSqlPagination
    {
        $perPage = $this->getPerPageFromRequest($request);
        $page    = $this->getPageFromRequest($request);
        
        return WithdrawalSqlPagination::create($perPage, ($page - 1) * $perPage);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return WithdrawalSqlCriteria
     */
    public function getCriteriaFromRequest(ServerRequestInterface $request): WithdrawalSqlCriteria
    {
        try {
            $createdAfter = $request->getQueryParam('created-after');
            $createdAfter = ($createdAfter !== null) ? new DateTime($createdAfter) : $createdAfter;
        } catch (Exception $exception) {
            $createdAfter = null;
        }
        
        return WithdrawalSqlCriteria::create($this->getFiltersFromRequest($request),
                                             $this->getSortingFromRequest($request),
                                             $createdAfter);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return WithdrawalFilters
     */
    private function getFiltersFromRequest(ServerRequestInterface $request): WithdrawalFilters
    {
        $filters = [];
        foreach ($request->getQueryParam('filter', []) as $attribute => $value) {
            $operation = (strpos($value, '*') !== false) ? 'like' : 'eq';
            if (strpos($value, '|') > 0) {
                $operation = substr($value, 0, strpos($value, '|'));
                $value     = substr($value, strpos($value, '|') + 1);
            }
            
            $filters[] = WithdrawalFilter::create($attribute, $value, $operation);
        }
        
        return WithdrawalFilters::create(...$filters);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return WithdrawalSortings
     */
    private function getSortingFromRequest(ServerRequestInterface $request): WithdrawalSortings
    {
        $sortings = [];
        if ($request->getQueryParam('sort') !== null) {
            foreach (explode(',', $request->getQueryParam('sort')) as $attribute) {
                $order      = (strpos($attribute, '-') === 0) ? 'desc' : 'asc';
                $sortings[] = WithdrawalSorting::create(ltrim($attribute, '+-'), $order);
            }
        }
        
        return WithdrawalSortings::create(...$sortings);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return string
     */
    public function getResourceUrlFromRequest(ServerRequestInterface $request): string
    {
        return $request->getUri()->getScheme() . '://' . $request->getUri()->getHost() . $request->getUri()->getPath();
    }
}